#!/usr/bin/env php
<?php
/**
 * GalliumHash System Health Dashboard
 * 
 * Provides comprehensive health status for all ecosystem components
 * Web-accessible: http://yourdomain.com/automation/system-health.php
 */

date_default_timezone_set('UTC');
header('Content-Type: text/html; charset=utf-8');

// Configuration
$projectRoot = __DIR__ . '/..';
$logDir = __DIR__ . '/logs';
$statusFile = __DIR__ . '/logs/system-status.json';

/**
 * Load environment variables
 */
function loadEnvironment() {
    $envPath = __DIR__ . '/../../.env';
    if (file_exists($envPath)) {
        $lines = file($envPath, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
        foreach ($lines as $line) {
            if (strpos($line, '=') !== false && !empty(trim($line)) && !str_starts_with($line, '#')) {
                list($key, $value) = explode('=', $line, 2);
                $_ENV[trim($key)] = trim($value);
            }
        }
    }
}

/**
 * Execute command with timeout
 */
function executeCommand($command, $timeout = 30) {
    $output = [];
    $returnCode = 0;
    
    try {
        $process = proc_open(
            $command,
            [
                0 => ['pipe', 'r'],
                1 => ['pipe', 'w'],
                2 => ['pipe', 'w']
            ],
            $pipes
        );
        
        if (is_resource($process)) {
            fclose($pipes[0]);
            
            $startTime = time();
            while (!feof($pipes[1])) {
                $output[] = fgets($pipes[1]);
                
                if (time() - $startTime > $timeout) {
                    proc_terminate($process);
                    break;
                }
            }
            
            fclose($pipes[1]);
            fclose($pipes[2]);
            $returnCode = proc_close($process);
        }
        
        return [
            'success' => $returnCode === 0,
            'output' => implode('', $output),
            'return_code' => $returnCode
        ];
        
    } catch (Exception $e) {
        return [
            'success' => false,
            'output' => '',
            'error' => $e->getMessage()
        ];
    }
}

/**
 * Check system health
 */
function checkSystemHealth() {
    global $projectRoot;
    
    $health = [
        'timestamp' => date('c'),
        'systems' => [],
        'overall_status' => 'unknown'
    ];
    
    $systems = [
        'metabuilder' => [
            'name' => 'MetaBuilder (Bitcoin Data)',
            'path' => 'automation/metabuilder-collector.php',
            'health_cmd' => 'health',
            'test_cmd' => 'test'
        ],
        'metaminer' => [
            'name' => 'MetaMiner (Intelligence)',
            'path' => 'automation/metaminer-processor.php', 
            'health_cmd' => 'health',
            'test_cmd' => 'test'
        ],
        'ethereum_gas' => [
            'name' => 'Ethereum GasCollector',
            'path' => 'automation/ethereum-gas-collector.php',
            'health_cmd' => 'health',
            'test_cmd' => 'test'
        ]
    ];
    
    $healthyCount = 0;
    
    foreach ($systems as $systemId => $system) {
        $systemHealth = [
            'name' => $system['name'],
            'status' => 'unknown',
            'last_check' => date('c'),
            'details' => []
        ];
        
        try {
            // Test system availability
            $testCmd = "cd $projectRoot && php " . $system['path'] . ' ' . $system['test_cmd'];
            $testResult = executeCommand($testCmd, 20);
            
            if ($testResult['success']) {
                $systemHealth['status'] = 'reachable';
                
                // Check health
                $healthCmd = "cd $projectRoot && php " . $system['path'] . ' ' . $system['health_cmd'];
                $healthResult = executeCommand($healthCmd, 15);
                
                if ($healthResult['success']) {
                    $healthData = json_decode($healthResult['output'], true);
                    if ($healthData) {
                        $systemHealth['details'] = $healthData;
                        $systemHealth['status'] = 'healthy';
                        $healthyCount++;
                    }
                }
            } else {
                $systemHealth['status'] = 'unreachable';
                $systemHealth['error'] = $testResult['error'] ?? 'Command failed';
            }
            
        } catch (Exception $e) {
            $systemHealth['status'] = 'error';
            $systemHealth['error'] = $e->getMessage();
        }
        
        $health['systems'][$systemId] = $systemHealth;
    }
    
    // Determine overall status
    if ($healthyCount === count($systems)) {
        $health['overall_status'] = 'excellent';
    } elseif ($healthyCount >= count($systems) * 0.7) {
        $health['overall_status'] = 'good';
    } elseif ($healthyCount > 0) {
        $health['overall_status'] = 'partial';
    } else {
        $health['overall_status'] = 'critical';
    }
    
    return $health;
}

/**
 * Check database connectivity
 */
function checkDatabase() {
    loadEnvironment();
    
    try {
        $dsn = "mysql:host=" . ($_ENV['DB_HOST'] ?? 'localhost') . 
               ";charset=utf8mb4";
        $username = $_ENV['DB_USERNAME'] ?? 'nvdaxcom_galliumhash_metabuilder_user';
        $password = $_ENV['DB_PASSWORD'] ?? '32TbgulnE7YOd2G0e5';
        
        $db = new PDO($dsn, $username, $password);
        $db->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
        
        // Test multiple databases
        $databases = [
            'nvdaxcom_galliumhash_metabuilder' => 'MetaBuilder Database',
            'nvdaxcom_galliumhash_metachain' => 'MetaChain Database',
            'nvdaxcom_galliumhash_consciousness_witness' => 'Consciousness Witness Database'
        ];
        
        $results = [];
        foreach ($databases as $dbName => $displayName) {
            try {
                $stmt = $db->query("USE $dbName");
                if ($stmt) {
                    // Try to get record count from a sample table
                    $stmt = $db->query("SELECT COUNT(*) as count FROM information_schema.tables WHERE table_schema = '$dbName'");
                    $result = $stmt->fetch(PDO::FETCH_ASSOC);
                    
                    $results[$dbName] = [
                        'status' => 'connected',
                        'display_name' => $displayName,
                        'tables' => $result['count']
                    ];
                }
            } catch (Exception $e) {
                $results[$dbName] = [
                    'status' => 'disconnected',
                    'display_name' => $displayName,
                    'error' => $e->getMessage()
                ];
            }
        }
        
        return [
            'status' => 'connected',
            'databases' => $results,
            'timestamp' => date('c')
        ];
        
    } catch (Exception $e) {
        return [
            'status' => 'error',
            'error' => $e->getMessage(),
            'timestamp' => date('c')
        ];
    }
}

/**
 * Get log file analysis
 */
function analyzeLogs() {
    global $logDir;
    
    $logFiles = [
        'master-scheduler' => 'Master Scheduler',
        'metabuilder-collector' => 'MetaBuilder Collector',
        'metaminer-processor' => 'MetaMiner Processor',
        'ethereum-gas-collector' => 'Ethereum Gas Collector'
    ];
    
    $logAnalysis = [];
    
    foreach ($logFiles as $fileId => $displayName) {
        $logFile = "$logDir/$fileId.log";
        $logStatus = [
            'name' => $displayName,
            'file' => $logFile,
            'exists' => false,
            'last_modified' => null,
            'last_entry' => null,
            'recent_errors' => 0,
            'size' => 0
        ];
        
        if (file_exists($logFile)) {
            $logStatus['exists'] = true;
            $logStatus['size'] = filesize($logFile);
            $logStatus['last_modified'] = date('c', filemtime($logFile));
            
            // Get last few lines
            $lastLines = [];
            $lines = file($logFile);
            $recentLines = array_slice($lines, -10);
            $lastEntry = end($recentLines);
            
            $logStatus['last_entry'] = trim($lastEntry);
            
            // Count recent errors (last 50 lines)
            $checkLines = array_slice($lines, -50);
            $logStatus['recent_errors'] = count(array_filter($checkLines, function($line) {
                return stripos($line, 'error') !== false || stripos($line, 'failed') !== false;
            }));
        }
        
        $logAnalysis[$fileId] = $logStatus;
    }
    
    return $logAnalysis;
}

/**
 * Render status indicator
 */
function renderStatus($status, $size = 'md') {
    $icons = [
        'excellent' => '🟢',
        'good' => '🟡', 
        'partial' => '🟠',
        'healthy' => '🟢',
        'reachable' => '🟡',
        'unknown' => '⚪',
        'unreachable' => '🔴',
        'error' => '🔴',
        'critical' => '🔴',
        'disconnected' => '🔴'
    ];
    
    $icon = $icons[$status] ?? '⚪';
    
    $sizeClass = match($size) {
        'lg' => 'text-2xl',
        'md' => 'text-lg',
        'sm' => 'text-sm',
        default => 'text-base'
    };
    
    return "<span class='$sizeClass'>$icon</span>";
}

/**
 * Main execution
 */
$action = $_GET['action'] ?? 'dashboard';

try {
    switch ($action) {
        case 'json':
            // Return JSON health data
            header('Content-Type: application/json');
            
            $health = checkSystemHealth();
            $db = checkDatabase();
            $logs = analyzeLogs();
            
            $response = [
                'health' => $health,
                'database' => $db,
                'logs' => $logs,
                'timestamp' => date('c')
            ];
            
            echo json_encode($response, JSON_PRETTY_PRINT);
            break;
            
        case 'api':
            // API endpoint for external monitoring
            header('Content-Type: application/json');
            
            $health = checkSystemHealth();
            
            $response = [
                'overall_status' => $health['overall_status'],
                'systems' => array_map(function($system) {
                    return [
                        'name' => $system['name'],
                        'status' => $system['status']
                    ];
                }, $health['systems']),
                'timestamp' => date('c')
            ];
            
            echo json_encode($response);
            break;
            
        default:
            // Health dashboard HTML
            $health = checkSystemHealth();
            $db = checkDatabase();
            $logs = analyzeLogs();
            
            ?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>GalliumHash System Health Dashboard</title>
    <style>
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            margin: 0;
            padding: 20px;
            background: linear-gradient(135deg, #1a1a2e, #16213e);
            color: #fff;
            min-height: 100vh;
        }
        .container {
            max-width: 1200px;
            margin: 0 auto;
        }
        .header {
            text-align: center;
            margin-bottom: 30px;
        }
        .header h1 {
            color: #00d4ff;
            margin: 0;
            font-size: 2.5rem;
        }
        .timestamp {
            color: #ccc;
            font-size: 0.9rem;
        }
        .status-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
            gap: 20px;
            margin-bottom: 30px;
        }
        .status-card {
            background: rgba(255, 255, 255, 0.1);
            border-radius: 12px;
            padding: 20px;
            backdrop-filter: blur(10px);
            border: 1px solid rgba(255, 255, 255, 0.2);
        }
        .status-card h3 {
            margin-top: 0;
            color: #00d4ff;
            display: flex;
            align-items: center;
            gap: 10px;
        }
        .system-list {
            list-style: none;
            padding: 0;
            margin: 0;
        }
        .system-item {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: 10px 0;
            border-bottom: 1px solid rgba(255, 255, 255, 0.1);
        }
        .system-item:last-child {
            border-bottom: none;
        }
        .log-item {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: 8px 0;
            font-family: 'Courier New', monospace;
            font-size: 0.9rem;
        }
        .log-item .status {
            font-weight: bold;
        }
        .refresh-btn {
            background: #00d4ff;
            color: #1a1a2e;
            border: none;
            padding: 12px 24px;
            border-radius: 8px;
            cursor: pointer;
            font-weight: bold;
            margin-bottom: 20px;
        }
        .refresh-btn:hover {
            background: #00b8e6;
        }
        .alert {
            background: rgba(255, 69, 0, 0.2);
            border: 1px solid rgba(255, 69, 0, 0.5);
            border-radius: 8px;
            padding: 15px;
            margin: 20px 0;
        }
        .success {
            background: rgba(0, 255, 0, 0.1);
            border: 1px solid rgba(0, 255, 0, 0.3);
            border-radius: 8px;
            padding: 15px;
            margin: 20px 0;
        }
        .details {
            font-size: 0.9rem;
            color: #ccc;
            margin-top: 10px;
        }
    </style>
    <script>
        function refreshPage() {
            window.location.reload();
        }
        
        function refreshData() {
            fetch('?action=json')
                .then(response => response.json())
                .then(data => {
                    console.log('System health:', data);
                    // You can add more sophisticated updates here
                    refreshPage();
                })
                .catch(error => {
                    console.error('Error refreshing data:', error);
                });
        }
    </script>
</head>
<body>
    <div class="container">
        <div class="header">
            <h1>🚀 GalliumHash System Health</h1>
            <div class="timestamp">Last Updated: <?= date('Y-m-d H:i:s T') ?></div>
            <button class="refresh-btn" onclick="refreshPage()">🔄 Refresh Status</button>
        </div>

        <div class="status-grid">
            <!-- Overall Status -->
            <div class="status-card">
                <h3><?= renderStatus($health['overall_status'], 'lg') ?> Overall System Status</h3>
                <div style="font-size: 2rem; font-weight: bold; color: <?= $health['overall_status'] === 'excellent' ? '#00ff00' : ($health['overall_status'] === 'good' ? '#ffeb3b' : '#ff4500') ?>">
                    <?= strtoupper($health['overall_status']) ?>
                </div>
                <div class="details">
                    <?= count(array_filter($health['systems'], fn($s) => $s['status'] === 'healthy')) ?>/<?= count($health['systems']) ?> systems healthy
                </div>
            </div>

            <!-- System Health -->
            <div class="status-card">
                <h3><?= renderStatus('good') ?> System Components</h3>
                <ul class="system-list">
                    <?php foreach ($health['systems'] as $system): ?>
                    <li class="system-item">
                        <span><?= htmlspecialchars($system['name']) ?></span>
                        <span><?= renderStatus($system['status']) ?></span>
                    </li>
                    <?php endforeach; ?>
                </ul>
            </div>

            <!-- Database Status -->
            <div class="status-card">
                <h3><?= renderStatus($db['status']) ?> Database Connections</h3>
                <ul class="system-list">
                    <?php if (isset($db['databases'])): ?>
                        <?php foreach ($db['databases'] as $dbName => $dbInfo): ?>
                        <li class="system-item">
                            <span><?= htmlspecialchars($dbInfo['display_name']) ?></span>
                            <span><?= renderStatus($dbInfo['status']) ?></span>
                        </li>
                        <?php endforeach; ?>
                    <?php else: ?>
                        <li class="system-item">
                            <span>Database Check Failed</span>
                            <span><?= renderStatus('error') ?></span>
                        </li>
                    <?php endif; ?>
                </ul>
            </div>

            <!-- Log Status -->
            <div class="status-card">
                <h3>📊 Log Files Status</h3>
                <ul class="system-list">
                    <?php foreach ($logs as $log): ?>
                    <li class="system-item">
                        <span><?= htmlspecialchars($log['name']) ?></span>
                        <span>
                            <?php if ($log['exists']): ?>
                                <?= renderStatus($log['recent_errors'] > 0 ? 'partial' : 'good') ?>
                            <?php else: ?>
                                <?= renderStatus('unknown') ?>
                            <?php endif; ?>
                        </span>
                    </li>
                    <?php endforeach; ?>
                </ul>
            </div>
        </div>

        <?php if ($health['overall_status'] === 'excellent'): ?>
        <div class="success">
            <strong>🎉 Excellent!</strong> All systems are running optimally. Your GalliumHash ecosystem is healthy and automated.
        </div>
        <?php elseif ($health['overall_status'] === 'critical'): ?>
        <div class="alert">
            <strong>⚠️ Critical Issues Detected!</strong> Multiple systems are not responding. Check the logs and cron jobs.
        </div>
        <?php endif; ?>

        <div class="status-card">
            <h3>📋 Quick Actions</h3>
            <div style="display: flex; gap: 10px; flex-wrap: wrap;">
                <a href="?action=test" style="text-decoration: none; color: inherit;">
                    <button style="background: #28a745; color: white; border: none; padding: 8px 16px; border-radius: 4px; cursor: pointer;">Run Tests</button>
                </a>
                <a href="?action=json" style="text-decoration: none; color: inherit;" target="_blank">
                    <button style="background: #17a2b8; color: white; border: none; padding: 8px 16px; border-radius: 4px; cursor: pointer;">View JSON</button>
                </a>
                <a href="CRON_SETUP.md" style="text-decoration: none; color: inherit;" target="_blank">
                    <button style="background: #6c757d; color: white; border: none; padding: 8px 16px; border-radius: 4px; cursor: pointer;">Cron Setup</button>
                </a>
            </div>
        </div>
    </div>
</body>
</html>
            <?php
            break;
    }
    
} catch (Exception $e) {
    if ($action === 'json' || $action === 'api') {
        header('Content-Type: application/json');
        echo json_encode([
            'error' => true,
            'message' => $e->getMessage(),
            'timestamp' => date('c')
        ], JSON_PRETTY_PRINT);
    } else {
        echo "<h1>Error</h1><p>" . htmlspecialchars($e->getMessage()) . "</p>";
    }
    exit(1);
}